<?php

declare(strict_types=1);

namespace Gls\GlsPoland\PrestaShop\Grid\Definition\Factory;

use Gls\GlsPoland\Controller\Admin\ConfigurationController;
use Gls\GlsPoland\Form\Type\CarrierTypeChoiceType;
use Gls\GlsPoland\PrestaShop\Grid\Action\Row\LegacyLinkRowAction;
use Gls\GlsPoland\Translation\TranslatorAwareTrait;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\RowActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\RowActionCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\Type\SubmitRowAction;
use PrestaShop\PrestaShop\Core\Grid\Action\Type\SimpleGridAction;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollection;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\ActionColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\DataColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\IdentifierColumn;
use PrestaShop\PrestaShop\Core\Grid\Definition\Factory\AbstractGridDefinitionFactory;
use PrestaShop\PrestaShop\Core\Grid\Filter\Filter;
use PrestaShop\PrestaShop\Core\Grid\Filter\FilterCollection;
use PrestaShop\PrestaShop\Core\Grid\Filter\FilterCollectionInterface;
use PrestaShop\PrestaShop\Core\Hook\HookDispatcherInterface;
use PrestaShopBundle\Event\Dispatcher\NullDispatcher;
use PrestaShopBundle\Form\Admin\Type\SearchAndResetType;
use PrestaShopBundle\Security\Voter\PageVoter;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;

final class CarrierGridDefinitionFactory extends AbstractGridDefinitionFactory implements FilterableGridDefinitionFactoryInterface
{
    use TranslatorAwareTrait;

    public const GRID_ID = 'glspoland_carrier';

    private $authorizationChecker;

    public function __construct(AuthorizationCheckerInterface $authorizationChecker, ?HookDispatcherInterface $hookDispatcher = null)
    {
        parent::__construct($hookDispatcher ?? new NullDispatcher());

        $this->authorizationChecker = $authorizationChecker;
    }

    protected function getId(): string
    {
        return self::GRID_ID;
    }

    public function getFilterId(): string
    {
        return $this->getId();
    }

    protected function getName(): string
    {
        return $this->trans('Carriers', [], 'Modules.Glspoland.Carrier');
    }

    protected function getColumns(): ColumnCollectionInterface
    {
        if (!class_exists(DataColumn::class)) {
            class_alias(\PrestaShop\PrestaShop\Core\Grid\Column\Type\DataColumn::class, DataColumn::class);
        }

        return (new ColumnCollection())
            ->add(
                (new IdentifierColumn('id_reference'))
                    ->setName($this->trans('Reference ID', [], 'Modules.Glspoland.Carrier'))
                    ->setOptions([
                        'identifier_field' => 'id_reference',
                    ])
            )
            ->add(
                (new DataColumn('type'))
                    ->setName($this->trans('Type', [], 'Modules.Glspoland.Carrier'))
                    ->setOptions([
                        'field' => 'type',
                    ])
            )
            ->add(
                (new DataColumn('name'))
                    ->setName($this->trans('Name', [], 'Admin.Global'))
                    ->setOptions([
                        'field' => 'name',
                    ])
            )
            ->add(
                (new DataColumn('countries'))
                    ->setName($this->trans('Countries', [], 'Modules.Glspoland.Carrier'))
                    ->setOptions([
                        'field' => 'countries',
                    ])
            )
            ->add(
                (new DataColumn('services'))
                    ->setName($this->trans('Additional services', [], 'Modules.Glspoland.Carrier'))
                    ->setOptions([
                        'field' => 'services',
                    ])
            )
            ->add(
                (new ActionColumn('actions'))
                    ->setName($this->trans('Actions', [], 'Admin.Global'))
                    ->setOptions([
                        'actions' => $this->getRowActions(),
                    ])
            );
    }

    protected function getFilters(): FilterCollectionInterface
    {
        return (new FilterCollection())
            ->add(
                (new Filter('id_reference', NumberType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search ID', [], 'Modules.Glspoland.Admin'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('id_reference')
            )
            ->add(
                (new Filter('type', CarrierTypeChoiceType::class))
                    ->setTypeOptions([
                        'placeholder' => $this->trans('All', [], 'Admin.Global'),
                        'required' => false,
                    ])
                    ->setAssociatedColumn('type')
            )
            ->add(
                (new Filter('name', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search name', [], 'Modules.Glspoland.Admin'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('name')
            )
            ->add(
                (new Filter('countries', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search country', [], 'Modules.Glspoland.Admin'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('countries')
            )
            ->add(
                (new Filter('actions', SearchAndResetType::class))
                    ->setTypeOptions([
                        'reset_route' => 'admin_common_reset_search_by_filter_id',
                        'reset_route_params' => [
                            'filterId' => self::GRID_ID,
                        ],
                        'redirect_route' => 'admin_gls_poland_config_carriers_index',
                    ])
                    ->setAssociatedColumn('actions')
            )
        ;
    }

    protected function getGridActions(): GridActionCollectionInterface
    {
        return (new GridActionCollection())
            ->add(
                (new SimpleGridAction('common_refresh_list'))
                    ->setName($this->trans('Refresh list', [], 'Admin.Advparameters.Feature'))
                    ->setIcon('refresh')
            )
            ->add(
                (new SimpleGridAction('common_show_query'))
                    ->setName($this->trans('Show SQL query', [], 'Admin.Actions'))
                    ->setIcon('code')
            )
            ->add(
                (new SimpleGridAction('common_export_sql_manager'))
                    ->setName($this->trans('Export to SQL Manager', [], 'Admin.Actions'))
                    ->setIcon('storage')
            );
    }

    private function getRowActions(): RowActionCollectionInterface
    {
        $collection = new RowActionCollection();

        if ($this->authorizationChecker->isGranted(PageVoter::UPDATE, ConfigurationController::CARRIERS_TAB_NAME)) {
            $collection->add(
                (new LegacyLinkRowAction('edit'))
                    ->setName($this->trans('Edit', [], 'Admin.Actions'))
                    ->setOptions([
                        'route' => 'AdminCarrierWizard',
                        'route_param_name' => 'id_carrier',
                        'route_param_field' => 'id_carrier',
                    ])
                    ->setIcon('edit')
            );
        }

        if ($this->authorizationChecker->isGranted(PageVoter::DELETE, ConfigurationController::TAB_NAME)) {
            $collection->add(
                (new SubmitRowAction('delete'))
                    ->setName($this->trans('Delete', [], 'Admin.Actions'))
                    ->setOptions([
                        'route' => 'admin_gls_poland_config_carriers_delete',
                        'route_param_name' => 'id',
                        'route_param_field' => 'id_reference',
                        'confirm_message' => $this->trans('Delete selected item?', [], 'Admin.Notifications.Warning'),
                    ])
                    ->setIcon('delete')
            );
        }

        return $collection;
    }
}
